using System;
using System.IO;
using System.Collections;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using System.Threading;

namespace SPStudio
{
	/// <summary>
	/// The main relay for data
	/// between objects.
	/// </summary>
	public class Workspace
	{
		#region Public Vars
		public bool creationCanceled = false;
		public double cameraFOV = 36.0;
		public double greatestX = 0.0;
		public double greatestY = 0.0;
		public double smallestY = 0.0;
		public double distance_to_camera = 0.0;
		public int vectorRows = 0;
		public int vectorCols = 0;
		
		/// <summary>
		/// Handle to the desktop
		/// </summary>
		public Desktop desktop = null;

		/// <summary>
		/// imageContainers = collection of handles to 
		/// ImageContainer instances associated with this workspace
		/// </summary>
		public ArrayList imageContainers = new ArrayList(1);

		/// <summary>
		///	shiftVoxels = collection of handles Voxel instances
		/// associated with this workspace
		/// </summary>
		public ArrayList shiftVoxels = new ArrayList(1);
		public ArrayList finalizedVoxels = new ArrayList(1);

		public Vector3D [,,] vectors;

		public bool closing = false;
		#endregion

		#region Private Vars
		/// <summary>
		/// Location of workspace resources
		/// </summary>
		private String location = "Workspaces\\";
		private String name = "Default";

		/// <summary>
		/// Handle to form to display list of
		/// images associated with the workspace,
		/// ordered by the associated degree
		/// </summary>
		private ImageList imageList = null;

		/// <summary>
		/// size = the resolution of the 
		/// photos in the workspace
		/// All must be the same
		/// </summary>
		private Size imageSize = new Size(0,0);

		private ShiftVoxelView shiftView = null;
		private ShiftForm shiftForm = null;

		/// <summary>
		/// model = collection of handles to Triangle3D instances
		/// associated with this workspace
		/// </summary>
		private ArrayList model = new ArrayList(1);
		#endregion

		#region Constructors
		/// <summary>
		/// Creates an instance of NewWorkspaceForm
		/// in order to gather the needed
		/// initialization information.
		/// Obtains a handle to the parent
		/// desktop.
		/// </summary>
		/// <param name="Parent"></param>
		public Workspace(Desktop Parent)
		{
			NewWorkspaceForm nwf = new NewWorkspaceForm();
			nwf.ShowDialog();

			if(nwf.creationCanceled)
			{
				creationCanceled = true;
				nwf.Close();
			}
			else
			{
				if(nwf.name.Trim().Equals(""))
				{
					location+="Default";
				}
				else
				{
					location+=nwf.name;
					name = nwf.name;
					cameraFOV = nwf.fov;
				}
				nwf.Close();
				try
				{
					Directory.CreateDirectory(location);

					desktop = Parent;
					desktop.reset();
					desktop.toggleAddImage(true);
					desktop.toggleSaveWkspc(true);
					desktop.toggleViewMenu(false);			
					desktop.toggleCloseWkspc(true);
					desktop.toggleShiftMenu(false);
					desktop.toggleLoadModelMenu(true);

					imageList = new ImageList(this);
					imageList.MdiParent = desktop;
				}
				catch(Exception)
				{
					MessageBox.Show("Unable to create directory\r\n"+location);
					creationCanceled = true;
				}				
			}
		}

		/// Constructor used if initialization
		/// data can been obtained
		/// through File I/O.
		/// Assumptions are made that the
		/// data doesn't need to be tested for
		/// validity since in order to be saved
		/// the data must have been validated
		/// previously.
		public Workspace(Desktop Parent, String File)
		{
			desktop = Parent;
			desktop.reset();
			desktop.toggleAddImage(true);
			desktop.toggleSaveWkspc(true);
			desktop.toggleViewMenu(false);			
			desktop.toggleCloseWkspc(true);
			desktop.toggleShiftMenu(false);
			desktop.toggleLoadModelMenu(true);

			imageList = new ImageList(this);
			imageList.MdiParent = desktop;

			FileHandler fh = new FileHandler();
			fh.setReadFile(File);

			location = fh.getLine().Trim();
			name = fh.getLine().Trim();
			cameraFOV = Convert.ToDouble(fh.getLine().Trim());
			distance_to_camera = Convert.ToDouble(fh.getLine().Trim());
			greatestX = Convert.ToDouble(fh.getLine().Trim());
			greatestY = Convert.ToDouble(fh.getLine().Trim());
			smallestY = Convert.ToDouble(fh.getLine().Trim());
			imageSize = new Size(Convert.ToInt32(fh.getLine().Trim()),Convert.ToInt32(fh.getLine().Trim()));

			while(!fh.isEndOfFile())
			{
				String lineIn = fh.getLine();//skip the "BEGIN_IMAGE" label in the file
				ArrayList groups = new ArrayList(1);
				ArrayList points = new ArrayList(1);
				String imageFile = "";
				double degree = 0.0;

				imageFile = fh.getLine().Trim();
				degree = Convert.ToDouble(fh.getLine().Trim());

				lineIn = fh.getLine();//skip the "BEGIN_GROUP" label in the file
				while(!lineIn.Equals("END_IMAGE"))
				{										
					lineIn = fh.getLine().Trim();
					points = new ArrayList(1);
					while(!lineIn.Equals("END_GROUP"))
					{
						points.Add(new Point(Convert.ToInt32(lineIn),Convert.ToInt32(fh.getLine().Trim())));
						points.Capacity++;
						lineIn = fh.getLine().Trim();
					}
					lineIn = fh.getLine().Trim();
					groups.Add(points);
					groups.Capacity++;
				}

				imageContainers.Add(new ImageContainer(this,imageFile,degree,groups));
				imageList.updateList(imageContainers);
			}
			fh.closeReader();

			try
			{
				fh.setReadFile(location+"\\"+name+".sft");

				vectorCols = Convert.ToInt32(fh.getLine().Trim());
				vectorRows = Convert.ToInt32(fh.getLine().Trim());
				vectors = new Vector3D[vectorCols,vectorRows,vectorCols];

				for(int z = 0; z<vectorCols; z++)
					for(int y = 0; y<vectorRows; y++)
						for(int x = 0; x<vectorCols; x++)
						{
							vectors[x,y,z] = new Vector3D(
							Convert.ToDouble(fh.getLine()),
							Convert.ToDouble(fh.getLine()),
							Convert.ToDouble(fh.getLine()),
							Convert.ToBoolean(fh.getLine()),
							Convert.ToBoolean(fh.getLine()),
							Convert.ToBoolean(fh.getLine()),
							Convert.ToBoolean(fh.getLine()));
						}
				fh.closeReader();

				int columns = vectorCols-1;
				int rows    = vectorRows-1;

				for(int zIndex=0; zIndex<columns; zIndex++)
					for(int yIndex=0; yIndex<rows; yIndex++)
						for(int xIndex=0; xIndex<columns; xIndex++)
						{
							shiftVoxels.Add(new Voxel(
								vectors[xIndex,yIndex,zIndex],
								vectors[xIndex+1,yIndex,zIndex],
								vectors[xIndex+1,yIndex+1,zIndex],
								vectors[xIndex,yIndex+1,zIndex],
								vectors[xIndex,yIndex,zIndex+1],
								vectors[xIndex+1,yIndex,zIndex+1],
								vectors[xIndex+1,yIndex+1,zIndex+1],
								vectors[xIndex,yIndex+1,zIndex+1]));
							shiftVoxels.Capacity++;
						}
				desktop.toggleShiftMenu(true);
			}
			catch(Exception)
			{
				//Failure means that there is no
				//sft file associated with the workspace
				//yet. This means that the algorithm
				//has yet to be run.
			}

			if(imageContainers.Count>0)
				desktop.toggleViewMenu(true);

			bool region1Covered = false;
			bool region2Covered = false;
			bool region3Covered = false;
			bool region4Covered = false;

			for(int containerIndex=0; containerIndex<imageContainers.Count; containerIndex++)
			{
				ImageContainer tic = (ImageContainer)imageContainers[containerIndex];
				if(((tic.degree>=0.0)&&(tic.degree<90.0))||(tic.degree==0.0))
					region1Covered = true;
				if((tic.degree>=90.0)&&(tic.degree<180.0))
					region2Covered = true;
				if((tic.degree>=180.0)&&(tic.degree<270.0))
					region3Covered = true;
				if((tic.degree>=270.0)&&(tic.degree<360.0))
					region4Covered = true;
			}
			//Verify that there are enough images to
			//account for all four of the regions.
			desktop.toggleRunAlgMenu(region1Covered&&region2Covered&&region3Covered&&region4Covered);
		}
		#endregion

		#region Load .mdl File
		/// <summary>
		/// Loads the .mdl file
		/// associated with the
		/// active workspace.
		/// </summary>
		public void loadMdl()
		{
			FileHandler fh = new FileHandler();

			try
			{
				fh.setReadFile(location+"\\"+name+".mdl");

				while(!fh.isEndOfFile())
				{
					Vector3D p1 = new Vector3D(Convert.ToDouble(fh.getLine().Trim()),Convert.ToDouble(fh.getLine().Trim()),Convert.ToDouble(fh.getLine().Trim()));
					Vector3D p2 = new Vector3D(Convert.ToDouble(fh.getLine().Trim()),Convert.ToDouble(fh.getLine().Trim()),Convert.ToDouble(fh.getLine().Trim()));
					Vector3D p3 = new Vector3D(Convert.ToDouble(fh.getLine().Trim()),Convert.ToDouble(fh.getLine().Trim()),Convert.ToDouble(fh.getLine().Trim()));
					Vector3D p4 = new Vector3D(Convert.ToDouble(fh.getLine().Trim()),Convert.ToDouble(fh.getLine().Trim()),Convert.ToDouble(fh.getLine().Trim()));
					Vector3D p5 = new Vector3D(Convert.ToDouble(fh.getLine().Trim()),Convert.ToDouble(fh.getLine().Trim()),Convert.ToDouble(fh.getLine().Trim()));
					Vector3D p6 = new Vector3D(Convert.ToDouble(fh.getLine().Trim()),Convert.ToDouble(fh.getLine().Trim()),Convert.ToDouble(fh.getLine().Trim()));
					Vector3D p7 = new Vector3D(Convert.ToDouble(fh.getLine().Trim()),Convert.ToDouble(fh.getLine().Trim()),Convert.ToDouble(fh.getLine().Trim()));
					Vector3D p8 = new Vector3D(Convert.ToDouble(fh.getLine().Trim()),Convert.ToDouble(fh.getLine().Trim()),Convert.ToDouble(fh.getLine().Trim()));
					
					finalizedVoxels.Add(new Voxel(p1,p2,p3,p4,p5,p6,p7,p8));
					finalizedVoxels.Capacity++;
				}
				new VoxelView(this);
			}
			catch(Exception)
			{
				MessageBox.Show("Model file not found");
			}
		}
		#endregion

		#region Save .mdl File
		/// <summary>
		/// Output the Finalized Voxels
		/// to a .mdl file.
		/// </summary>
		public void saveMdl()
		{
			FileHandler fh = new FileHandler();
			fh.setWriteFile(location+"\\"+name+".mdl");

			IEnumerator voxelWalk = finalizedVoxels.GetEnumerator();
			while(voxelWalk.MoveNext())
			{
				Voxel v = (Voxel)voxelWalk.Current;
				fh.writeLine(v.p1.x.ToString());
				fh.writeLine(v.p1.y.ToString());
				fh.writeLine(v.p1.z.ToString());

				fh.writeLine(v.p2.x.ToString());
				fh.writeLine(v.p2.y.ToString());
				fh.writeLine(v.p2.z.ToString());

				fh.writeLine(v.p3.x.ToString());
				fh.writeLine(v.p3.y.ToString());
				fh.writeLine(v.p3.z.ToString());

				fh.writeLine(v.p4.x.ToString());
				fh.writeLine(v.p4.y.ToString());
				fh.writeLine(v.p4.z.ToString());

				fh.writeLine(v.p5.x.ToString());
				fh.writeLine(v.p5.y.ToString());
				fh.writeLine(v.p5.z.ToString());

				fh.writeLine(v.p6.x.ToString());
				fh.writeLine(v.p6.y.ToString());
				fh.writeLine(v.p6.z.ToString());

				fh.writeLine(v.p7.x.ToString());
				fh.writeLine(v.p7.y.ToString());
				fh.writeLine(v.p7.z.ToString());

				fh.writeLine(v.p8.x.ToString());
				fh.writeLine(v.p8.y.ToString());
				fh.writeLine(v.p8.z.ToString());
			}
			fh.closeWriter();
		}
		#endregion

		#region Save .wks File
		/// <summary>
		/// Save the workspace as
		/// a .wks file.
		/// </summary>
		public void save()
		{
			FileHandler fh = new FileHandler();

			fh.setWriteFile(location+"\\"+name+".wks");

			fh.writeLine(location);
			fh.writeLine(name);
			fh.writeLine(cameraFOV.ToString());
			fh.writeLine(distance_to_camera.ToString());
			fh.writeLine(greatestX.ToString());
			fh.writeLine(greatestY.ToString());
			fh.writeLine(smallestY.ToString());
			fh.writeLine(imageSize.Width.ToString());
			fh.writeLine(imageSize.Height.ToString());
			IEnumerator imageWalk = imageContainers.GetEnumerator();
			while(imageWalk.MoveNext())
			{
				fh.writeLine("BEGIN_IMAGE");
				ImageContainer ic = (ImageContainer)imageWalk.Current;
				fh.writeLine(ic.imageFile);
				double tempDeg = ic.degree;
				fh.writeLine(tempDeg.ToString());
				IEnumerator groupWalk = ic.groups.GetEnumerator();
				while(groupWalk.MoveNext())
				{
					fh.writeLine("BEGIN_GROUP");
					IEnumerator pointWalk = ((ArrayList)groupWalk.Current).GetEnumerator();
					while(pointWalk.MoveNext())
					{
						Point p = (Point)pointWalk.Current;
						fh.writeLine(p.X.ToString());
						fh.writeLine(p.Y.ToString());
					}
					fh.writeLine("END_GROUP");
				}
				fh.writeLine("END_IMAGE");
			}
			fh.closeWriter();

			if(shiftVoxels.Count>0)
			{
				fh.setWriteFile(location+"\\"+name+".sft");

				fh.writeLine(vectorCols.ToString());
				fh.writeLine(vectorRows.ToString());

				for(int z = 0; z<vectorCols; z++)
					for(int y = 0; y<vectorRows; y++)
						for(int x = 0; x<vectorCols; x++)
						{
							Vector3D v = vectors[x,y,z];
							fh.writeLine(v.x.ToString());
							fh.writeLine(v.y.ToString());
							fh.writeLine(v.z.ToString());
							fh.writeLine(v.in0to90.ToString());
							fh.writeLine(v.in90to180.ToString());
							fh.writeLine(v.in180to270.ToString());
							fh.writeLine(v.in270to360.ToString());
						}
				fh.closeWriter();
			}
		}
		#endregion

		#region Close the workspace
		/// <summary>
		/// Closes all components associated
		/// with the workspace as well
		/// as the workspace itself.
		/// </summary>
		public void close()
		{
			closing = true;			
			//Close all components
			//associated with the workspace
			if(imageList!=null)
				imageList.Close();
			if(shiftView!=null)
				shiftView.Close();
			if(shiftForm!=null)
				shiftForm.Close();
			IEnumerator imageWalk = imageContainers.GetEnumerator();
			while(imageWalk.MoveNext())
			{
				removeImage((ImageContainer)imageWalk.Current);
				imageWalk = imageContainers.GetEnumerator();
			}
		}
		#endregion

		#region View
		/// <summary>
		/// Trigger the ImageList to be shown
		/// </summary>
		public void viewMenu()
		{
			imageList.Show();
			imageList.Focus();
		}
		#endregion

		#region Image Count
		/// <summary>
		/// Return the number of
		/// images associated with
		/// workspace.
		/// </summary>
		/// <returns></returns>
		public int imageCount()
		{
			return imageContainers.Count;
		}
		#endregion

		#region Workspace Size Verify and Set
		/// <summary>
		/// Attempt to set the workspace size,
		/// if it has already been set then
		/// nothing will happen
		/// </summary>
		/// <param name="ImageSize"></param>
		public void setImageSize(Size ImageSize)
		{
			if((imageSize.Height==0)||(imageSize.Width==0))
			{
				greatestX = (double)(-1)*ImageSize.Width;
				greatestY = (double)(-1)*ImageSize.Height;
				smallestY = (double)ImageSize.Height;
				imageSize = ImageSize;
			}
		}

		/// <summary>
		/// Verify that the param Size
		/// is the same as the workspace size
		/// </summary>
		/// <param name="inputSize"></param>
		/// <returns></returns>
		public bool verifySize(Size inputSize)
		{
			return ((inputSize.Height==imageSize.Height)&&(inputSize.Width==imageSize.Width));
		}

		/// <summary>
		/// Return the current workspace's
		/// size setting.
		/// </summary>
		/// <returns></returns>
		public Size getSize()
		{
			if((imageSize.Height==0)||(imageSize.Width==0)) return new Size(0,0);
			else return imageSize;
		}
		#endregion

		#region Add and Remove Images
		/// <summary>
		/// Attempt to add an image to
		/// the workspace and if enough
		/// images are added then
		/// allow for the algorithm to
		/// be run.
		/// </summary>
		public void addImage()
		{
			ImageContainer ic = new ImageContainer(this);
			if(ic.creationCanceled)
			{
				ic.Dispose();
			}
			else
			{
				desktop.toggleViewMenu(true);
				imageContainers.Add(ic);
				imageContainers.Capacity++;
				imageList.updateList(imageContainers);
				ic.MdiParent = desktop;
				ic.Show();

				bool region1Covered = false;
				bool region2Covered = false;
				bool region3Covered = false;
				bool region4Covered = false;

				for(int containerIndex=0; containerIndex<imageContainers.Count; containerIndex++)
				{
					ImageContainer tic = (ImageContainer)imageContainers[containerIndex];
					if(((tic.degree>=0.0)&&(tic.degree<90.0))||(tic.degree==0.0))
						region1Covered = true;
					if((tic.degree>=90.0)&&(tic.degree<180.0))
						region2Covered = true;
					if((tic.degree>=180.0)&&(tic.degree<270.0))
						region3Covered = true;
					if((tic.degree>=270.0)&&(tic.degree<360.0))
						region4Covered = true;
				}
				desktop.toggleRunAlgMenu(region1Covered&&region2Covered&&region3Covered&&region4Covered);
			}
		}

		/// <summary>
		/// Remove an image from
		/// the workspace and if enough
		/// images are removed then
		/// do not allow the algorithm to
		/// be run.
		/// </summary>
		/// <param name="ic"></param>
		public void removeImage(ImageContainer ic)
		{
			imageContainers.Remove(ic);
			ic.removed = true;
			ic.Close();
			imageContainers.Capacity--;
			imageList.updateList(imageContainers);
			if(imageContainers.Count==0)
			{
				desktop.toggleViewMenu(false);
			}

			bool region1Covered = false;
			bool region2Covered = false;
			bool region3Covered = false;
			bool region4Covered = false;

			for(int containerIndex=0; containerIndex<imageContainers.Count; containerIndex++)
			{
				ImageContainer tic = (ImageContainer)imageContainers[containerIndex];
				if(((tic.degree>=0.0)&&(tic.degree<90.0))||(tic.degree==0.0))
					region1Covered = true;
				if((tic.degree>=90.0)&&(tic.degree<180.0))
					region2Covered = true;
				if((tic.degree>=180.0)&&(tic.degree<270.0))
					region3Covered = true;
				if((tic.degree>=270.0)&&(tic.degree<360.0))
					region4Covered = true;
			}
			desktop.toggleRunAlgMenu(region1Covered&&region2Covered&&region3Covered&&region4Covered);
		}
		#endregion

		#region Shift View/Form Controls
		/// <summary>
		/// Call the Shift View's Refresh method.
		/// </summary>
		public void refreshShiftView()
		{
			if(shiftView!=null)
				shiftView.Refresh();
		}

		/// <summary>
		/// Revalidate the shift view
		/// and shift form.
		/// </summary>
		public void activateShift()
		{
			if(shiftView!=null)
				shiftView.Close();
			if(shiftForm!=null)
				shiftForm.Close();
			shiftView = new ShiftVoxelView(this);
			shiftForm = new ShiftForm(this);
		}
		#endregion

		#region Algorithm
		/// <summary>
		/// Gather the 3D triangles from the image
		/// containers, send them to the algorithm.
		/// Retrieve the generated data, if any.
		/// </summary>
		public void runAlgorithm()
		{
			//Collect all triangles for the algorithm
			ArrayList region1TriangleGroups = new ArrayList(1);
			ArrayList region2TriangleGroups = new ArrayList(1);
			ArrayList region3TriangleGroups = new ArrayList(1);
			ArrayList region4TriangleGroups = new ArrayList(1);

			bool missingData = false;
			String containersMissingData = "";

			for(int containerIndex=0; (containerIndex<imageContainers.Count); containerIndex++)
			{
				ImageContainer ic = (ImageContainer)imageContainers[containerIndex];

				ArrayList triangleSubset = ic.getTriangles();

				if(triangleSubset.Count>0)
				{
					if(distance_to_camera == 0.0)
						distance_to_camera = ic.distance_to_camera;
					if((ic.degree>=0.0)&&(ic.degree<=90.0))
					{
						region1TriangleGroups.Add(triangleSubset);
						region1TriangleGroups.Capacity++;
					}
					if((ic.degree>=90.0)&&(ic.degree<=180.0))
					{
						region2TriangleGroups.Add(triangleSubset);
						region2TriangleGroups.Capacity++;
					}
					if((ic.degree>=180.0)&&(ic.degree<=270.0))
					{
						region3TriangleGroups.Add(triangleSubset);
						region3TriangleGroups.Capacity++;
					}
					if((ic.degree>=270.0)&&(ic.degree<=360.0)||(ic.degree==0.0))
					{
						region4TriangleGroups.Add(triangleSubset);
						region4TriangleGroups.Capacity++;
					}

					if(ic.greatestX>greatestX)
						greatestX = ic.greatestX;
					if(ic.greatestY>greatestY)
						greatestY = ic.greatestY;
					if(ic.smallestY<smallestY)
						smallestY = ic.smallestY;
				}
				else
				{
					containersMissingData += ic.ToString() + "\r\n";
					missingData = true;
				}
			}
			if(!missingData)
			{
				//Run algorithm and get shiftVoxels
				VRMCAlg alg = new VRMCAlg(this, region1TriangleGroups, region2TriangleGroups,
					region3TriangleGroups, region4TriangleGroups);
		
				alg.ShowDialog();

				shiftVoxels = alg.getShiftVoxels();
				vectors = alg.getVectors();
				vectorRows = alg.getVectorRows();
				vectorCols = alg.getVectorCols();

				alg.Close();
		
				if(shiftVoxels.Count>0)
				{
					desktop.toggleShiftMenu(true);
					activateShift();
				}
				else
				{
					MessageBox.Show("No shift voxels were generated.");
				}
			}
			else
			{
				MessageBox.Show("The following images are missing data \r\n"+containersMissingData);
			}
		}
		#endregion
	}
}
